import fundingCsv from "./ai-funding-statistics.csv?raw";
import perceptionCsv from "./consumer-perception.csv?raw";
import regulationCsv from "./support-ai-regulations.csv?raw";
import employmentCsv from "./ai-employment-trends.csv?raw";
import europeCsv from "./europe-ai-enterprise-adoption.csv?raw";
import chatgptCsv from "./chatgpt-business-operations.csv?raw";
import nlpCsv from "./nlp-stats.csv?raw";
import securityCsv from "./ai-security-stats.csv?raw";
import adoptionIndustriesCsv from "./ai-adoption-industries.csv?raw";
import customerServiceCsv from "./ai-customer-service.csv?raw";
import everydayLiveCsv from "./ai-everyday-live.csv?raw";
import jobStatsCsv from "./ai-job-stats.csv?raw";
import marketSizeCsv from "./ai-market-size-statistics.csv?raw";
import marketingCsv from "./ai-marketing-statistics.csv?raw";
import sensorMarketCsv from "./ai-sensor-market-size.csv?raw";
import userTrustCsv from "./ai-user-trust.csv?raw";
import ecommerceCsv from "./ecommerce-retail-stats.csv?raw";
import editorsChoiceCsv from "./editors-choice.csv?raw";

// Color palette
const colors = {
  blue: "#3b82f6",
  red: "#ef4444",
  green: "#22c55e",
  purple: "#a855f7",
  orange: "#f97316",
  cyan: "#06b6d4",
  pink: "#ec4899",
  yellow: "#eab308",
  indigo: "#6366f1",
  teal: "#14b8a6",
};

type CsvRow = Record<string, string>;

function parseCSV(csv: string): CsvRow[] {
  const lines = csv.trim().split("\n");
  const headers = lines[0].split(",");
  return lines.slice(1).filter(line => line.trim()).map((line) => {
    const values = line.split(",");
    return headers.reduce(
      (obj, h, i) => ({ ...obj, [h.trim()]: values[i]?.trim() || "" }),
      {} as CsvRow
    );
  });
}

function findRow(rows: CsvRow[], statistic: string): CsvRow | undefined {
  return rows.find((r) => r.Statistic?.toLowerCase().includes(statistic.toLowerCase()));
}

function getValue(row: CsvRow | undefined): number {
  if (!row) return 0;
  return parseFloat(row.Value) || 0;
}

// Parse all CSVs
const fundingRows = parseCSV(fundingCsv);
const perceptionRows = parseCSV(perceptionCsv);
const regulationRows = parseCSV(regulationCsv);
const employmentRows = parseCSV(employmentCsv);
const europeRows = parseCSV(europeCsv);
const chatgptRows = parseCSV(chatgptCsv);
const nlpRows = parseCSV(nlpCsv);
const securityRows = parseCSV(securityCsv);

// Chart 1: Funding Regional
export const fundingRegionalData = [
  {
    region: "United States",
    companies: getValue(findRow(fundingRows, "US newly funded")),
    fill: colors.blue,
  },
  {
    region: "European Union",
    companies: getValue(findRow(fundingRows, "EU newly funded")),
    fill: colors.red,
  },
  {
    region: "United Kingdom",
    companies: getValue(findRow(fundingRows, "UK newly funded")),
    fill: colors.green,
  },
];

// Chart 2: AI Market Projection
const marketValue2023 = getValue(findRow(fundingRows, "market value 2023"));
const marketValue2030Row = findRow(fundingRows, "market projected 2030");
const marketValue2030 = marketValue2030Row?.Unit?.includes("trillion")
  ? getValue(marketValue2030Row) * 1000
  : getValue(marketValue2030Row);

export const marketProjectionData = [
  { year: "2023", value: marketValue2023, fill: colors.indigo },
  { year: "2030 (Projected)", value: marketValue2030, fill: colors.purple },
];

// Chart 3: Perception Concerns (Pie)
export const perceptionConcernsData = [
  {
    name: "Inaccuracy Concerns",
    value: getValue(findRow(perceptionRows, "inaccuracy")),
    fill: colors.red,
  },
  {
    name: "IP Violation Worries",
    value: getValue(findRow(perceptionRows, "IP violations")),
    fill: colors.yellow,
  },
  {
    name: "Security Concerns",
    value: getValue(findRow(perceptionRows, "security")),
    fill: colors.blue,
  },
];

// Chart 4: Consumer Sentiment
export const consumerSentimentData = [
  {
    sentiment: "Feel AI is Overhyped",
    percentage: getValue(findRow(perceptionRows, "overhyped")),
    fill: colors.orange,
  },
  {
    sentiment: "Trust Has Grown Since GenAI",
    percentage: getValue(findRow(perceptionRows, "trust")),
    fill: colors.teal,
  },
];

// Chart 5: AI Regulation
export const regulationData = [
  {
    demand: "Want More AI Assurance Investment",
    percentage: getValue(findRow(regulationRows, "invest in AI assurance")),
    fill: colors.purple,
  },
  {
    demand: "Demand Transparency",
    percentage: getValue(findRow(regulationRows, "transparency")),
    fill: colors.cyan,
  },
];

// Chart 6: Employment Impact
export const employmentData = [
  {
    category: "By 2025",
    created: getValue(findRow(employmentRows, "jobs created")),
    eliminated: getValue(findRow(employmentRows, "eliminated")),
  },
];

// Chart 7: Europe AI Adoption
const europeCountries = ["Denmark", "Sweden", "Belgium", "Bulgaria", "Poland", "Romania"];
export const europeAdoptionData = europeCountries
  .map((country) => {
    const row = europeRows.find((r) => r.Region === country || r.Statistic?.includes(country));
    return {
      country,
      adoption: getValue(row),
    };
  })
  .sort((a, b) => b.adoption - a.adoption);

// Chart 8: ChatGPT Business Uses
const chatgptUseCases = [
  { key: "customer responses", label: "Customer Responses" },
  { key: "content generation", label: "Content Generation" },
  { key: "personalize", label: "Personalization" },
  { key: "web traffic", label: "Drive Web Traffic" },
  { key: "streamlin", label: "Streamline Processes" },
  { key: "summariz", label: "Summarizing Info" },
  { key: "decision", label: "Decision Making" },
];

export const chatgptUsesData = chatgptUseCases.map(({ key, label }) => ({
  useCase: label,
  percentage: getValue(findRow(chatgptRows, key)),
}));

// Chart 9: Voice Assistants (Pie)
export const voiceAssistantsData = [
  {
    name: "Amazon Alexa",
    value: getValue(findRow(nlpRows, "Alexa")),
    fill: "#FF9900",
  },
  {
    name: "Apple Siri",
    value: getValue(findRow(nlpRows, "Siri")),
    fill: "#A2AAAD",
  },
  {
    name: "Google Assistant",
    value: getValue(findRow(nlpRows, "Google")),
    fill: "#4285F4",
  },
];

// Chart 10: AI Security
export const securityData = [
  {
    metric: "Using AI for Fraud Detection",
    percentage: getValue(findRow(securityRows, "fraud")),
    fill: colors.green,
  },
  {
    metric: "Threat Detection Improvement",
    percentage: getValue(findRow(securityRows, "threat")),
    fill: colors.blue,
  },
];

export { colors };

// Raw CSV files for source download
export const rawCsvFiles: Record<string, string> = {
  "ai-funding-statistics.csv": fundingCsv,
  "consumer-perception.csv": perceptionCsv,
  "support-ai-regulations.csv": regulationCsv,
  "ai-employment-trends.csv": employmentCsv,
  "europe-ai-enterprise-adoption.csv": europeCsv,
  "chatgpt-business-operations.csv": chatgptCsv,
  "nlp-stats.csv": nlpCsv,
  "ai-security-stats.csv": securityCsv,
  "ai-adoption-industries.csv": adoptionIndustriesCsv,
  "ai-customer-service.csv": customerServiceCsv,
  "ai-everyday-live.csv": everydayLiveCsv,
  "ai-job-stats.csv": jobStatsCsv,
  "ai-market-size-statistics.csv": marketSizeCsv,
  "ai-marketing-statistics.csv": marketingCsv,
  "ai-sensor-market-size.csv": sensorMarketCsv,
  "ai-user-trust.csv": userTrustCsv,
  "ecommerce-retail-stats.csv": ecommerceCsv,
  "editors-choice.csv": editorsChoiceCsv,
};

// Chart metadata for combined export
export const chartMetadata = [
  { id: "funding-regional", title: "Newly Funded AI Companies by Region (2022)" },
  { id: "market-projection", title: "Global AI Market Value: 2023 vs 2030" },
  { id: "perception-concerns", title: "Primary Concerns Preventing AI Adoption" },
  { id: "consumer-sentiment", title: "Consumer Sentiment on AI" },
  { id: "regulation", title: "Public Demands for AI Regulation" },
  { id: "employment", title: "AI Impact on Global Employment by 2025" },
  { id: "europe-adoption", title: "AI Adoption Rates Across EU Countries (2024)" },
  { id: "chatgpt-uses", title: "Anticipated ChatGPT Business Use Cases" },
  { id: "voice-assistants", title: "US Voice Assistant Users (2022)" },
  { id: "security", title: "AI in Cybersecurity" },
];
